/*         ______   ___    ___ 
 *        /\  _  \ /\_ \  /\_ \ 
 *        \ \ \L\ \\//\ \ \//\ \      __     __   _ __   ___ 
 *         \ \  __ \ \ \ \  \ \ \   /'__`\ /'_ `\/\`'__\/ __`\
 *          \ \ \/\ \ \_\ \_ \_\ \_/\  __//\ \L\ \ \ \//\ \L\ \
 *           \ \_\ \_\/\____\/\____\ \____\ \____ \ \_\\ \____/
 *            \/_/\/_/\/____/\/____/\/____/\/___L\ \/_/ \/___/
 *                                           /\____/
 *                                           \_/__/
 *
 *      DirectDraw windowed gfx driver.
 *
 *      By Isaac Cruz.
 *
 *      See readme.txt for copyright information.
 */

#include "wddraw.h"


static struct BITMAP *gfx_directx_create_video_bitmap_win(int width, int height);
static void gfx_directx_destroy_video_bitmap_win(struct BITMAP *bitmap);
static void setup_retrace_proc (void);
static int wnd_set_windowed_coop(void);
static void create_offscreen (int w, int h, int color_depth);
static int verify_color_depth (int color_depth);
static int gfx_directx_show_video_bitmap_win(struct BITMAP *bitmap);
static struct BITMAP *init_directx_win(int w, int h, int v_w, int v_h, int color_depth);
static void gfx_directx_win_exit(struct BITMAP *b);


GFX_DRIVER gfx_directx_win =
{
   GFX_DIRECTX_WIN,
   empty_string,
   empty_string,
   "DirectDraw window",
   init_directx_win,
   gfx_directx_win_exit,
   NULL,                        // AL_METHOD(int, scroll, (int x, int y)); 
   gfx_directx_sync,
   gfx_directx_set_palette,
   NULL,                        // AL_METHOD(int, request_scroll, (int x, int y));
   NULL,                        // gfx_directx_poll_scroll,
   NULL,                        // AL_METHOD(void, enable_triple_buffer, (void));
   gfx_directx_create_video_bitmap,
   gfx_directx_destroy_video_bitmap,
   gfx_directx_show_video_bitmap_win,
   gfx_directx_show_video_bitmap_win,
   gfx_directx_create_system_bitmap,
   gfx_directx_destroy_system_bitmap,
   NULL,                        // AL_METHOD(int, set_mouse_sprite, (struct BITMAP *sprite, int xfocus, int yfocus));
   NULL,                        // AL_METHOD(int, show_mouse, (struct BITMAP *bmp, int x, int y));
   NULL,                        // AL_METHOD(void, hide_mouse, (void));
   NULL,                        // AL_METHOD(void, move_mouse, (int x, int y));
   NULL,                        // AL_METHOD(void, drawing_mode, (void));
   NULL,                        // AL_METHOD(void, save_video_state, (void*));
   NULL,                        // AL_METHOD(void, restore_video_state, (void*));
   0, 0,                        // int w, h;                     /* physical (not virtual!) screen size */
   TRUE,                        // int linear;                   /* true if video memory is linear */
   0,                           // long bank_size;               /* bank size, in bytes */
   0,                           // long bank_gran;               /* bank granularity, in bytes */
   0,                           // long vid_mem;                 /* video memory size, in bytes */
   0,                           // long vid_phys_base;           /* physical address of video memory */
};

static int pixel_match[] = { 8, 15, 16, 24, 24, 32, 32, 0 };

DDPIXELFORMAT pixel_format[] = {
   /* 8-bit */
   {sizeof(DDPIXELFORMAT), DDPF_RGB | DDPF_PALETTEINDEXED8, 0, 8, 0, 0, 0, 0},
   /* 16-bit RGB 5:5:5 */
   {sizeof(DDPIXELFORMAT), DDPF_RGB, 0, 16, 0x7C00, 0x03e0, 0x001F, 0},
   /* 16-bit RGB 5:6:5 */
   {sizeof(DDPIXELFORMAT), DDPF_RGB, 0, 16, 0xF800, 0x07e0, 0x001F, 0},
   /* 24-bit RGB */
   {sizeof(DDPIXELFORMAT), DDPF_RGB, 0, 24, 0xFF0000, 0x00FF00, 0x0000FF, 0},
   /* 24-bit BGR */
   {sizeof(DDPIXELFORMAT), DDPF_RGB, 0, 24, 0x0000FF, 0x00FF00, 0xFF0000, 0},
   /* 32-bit RGB */
   {sizeof(DDPIXELFORMAT), DDPF_RGB, 0, 32, 0xFF0000, 0x00FF00, 0x0000FF, 0},
   /* 32-bit BGR */
   {sizeof(DDPIXELFORMAT), DDPF_RGB, 0, 32, 0x0000FF, 0x00FF00, 0xFF0000, 0}
};

LPDDPIXELFORMAT dd_pixelformat = NULL;  /* pixel format used */

/* offscreen surface that will be blitted to the window:*/
LPDIRECTDRAWSURFACE dd_offscreen = NULL;
LPDIRECTDRAWSURFACE pseudo_screen = NULL;  /* for page-flipping */
BOOL app_active = TRUE;
RECT window_rect;
BOOL same_color_depth;
void (*update_window) (void);
static int desktop_depth = 0;   /* global variable */



/* handle_window_size_win:
 *  updates window_rect if window is moved or resized
 */
void handle_window_size_win(void)
{
   /* update window_rect */
   GetClientRect(allegro_wnd, &window_rect);
   ClientToScreen(allegro_wnd, (LPPOINT)&window_rect);
   ClientToScreen(allegro_wnd, (LPPOINT)&window_rect + 1);
}


/* update_window
 * function synced with the vertical refresh
 */
void update_window_hw (void)
{
   /* blit offscreen backbuffer to the window */
   _enter_gfx_critical();
   IDirectDrawSurface_Blt(dd_prim_surface,
			  &window_rect,
			  pseudo_screen, NULL,
			  0, NULL);
   _exit_gfx_critical();
}

/* update_window_ex
 * same as update_window, but performs color conversion
 */
void update_window_32_to_16 (void)
{
   DDSURFACEDESC src_desc, dest_desc;
   HRESULT hr;

   _enter_gfx_critical();
   src_desc.dwSize = sizeof(src_desc);
   src_desc.dwFlags = 0;
   dest_desc.dwSize = sizeof(dest_desc);
   dest_desc.dwFlags = 0;
   hr = IDirectDrawSurface_Lock(dd_prim_surface, &window_rect, &dest_desc, 0, NULL);
   if (FAILED(hr)) goto exit;
   hr = IDirectDrawSurface_Lock(pseudo_screen, NULL, &src_desc, 0, NULL);
   if (FAILED(hr)) {
      IDirectDrawSurface_Unlock(dd_prim_surface, NULL);
      goto exit;
   }

   /* 32 bit to 16 bit conversion */
   _asm {
      ; init register values

      mov eax, 0000FC00h   ; mask for green color
      movd mm3, eax
      punpckldq mm3, mm3   ; mm3 = 0x0000FC000000FC00
      mov eax, 00F80000h   ; mask for red color
      movd mm4, eax
      punpckldq mm4, mm4   ; mm4 = 0x00F8000000F80000
      mov eax, 000000F8h   ; mask for blue color
      movd mm5, eax
      punpckldq mm5, mm5   ; mm5 = 0x000000F8000000F8

      mov eax, gfx_driver
      mov edx, [eax]GFX_DRIVER.w     ; edx = SCREEN_W
      mov ecx, [eax]GFX_DRIVER.h     ; ecx = SCREEN_H

      lea eax, src_desc
      mov esi, [eax]DDSURFACEDESC.lPitch
      mov eax, [eax]DDSURFACEDESC.lpSurface   ; eax = src_desc.lpSurface
      shl edx, 2
      sub esi, edx     ; esi = (src_desc.lPitch) - (SCREEN_W * 4)

      lea ebx, dest_desc
      shr edx, 1       ; edx = SCREEN_W * 2
      mov edi, [ebx]DDSURFACEDESC.lPitch
      mov ebx, [ebx]DDSURFACEDESC.lpSurface   ; ebx = dest_desc.lpSurface
      sub edi, edx     ; edi = (dest_desc.lPitch) - (SCREEN_W * 2)
      shr edx, 2       ; edx = SCREEN_W / 2

      ; so we have:
      ; eax = src_desc.lpSurface
      ; ebx = dest_desc.lpSurface
      ; ecx = SCREEN_H
      ; edx = SCREEN_W / 2
      ; esi = offset from the end of a line to the beginning of the next
      ; edi = same as esi, but for the dest bitmap

      push ebp
   next_line:
      mov ebp, 0      ; (better than xor ebp, ebp)

   next_block:
      movq mm0, [eax]
      movq mm1, mm0
      movq mm2, mm0
      pand mm0, mm5
      psrld mm0, 3
      pand mm1, mm3
      psrld mm1, 5
      por mm0, mm1
      add eax, 8
      pand mm2, mm4
      psrld mm2, 8
      por mm0, mm2
      movq mm6, mm0
      psrlq mm0, 16
      por mm6, mm0
      movd [ebx], mm6
      add ebx, 4

      inc ebp
      cmp ebp, edx
      jb next_block

      add eax, esi
      add ebx, edi
      sub ecx, 1
      jnz next_line

      pop ebp
      emms
   }


/*   src = src_desc.lpSurface;
   dest = dest_desc.lpSurface;
   src_i = 0;
   dest_i = 0;
   w = SCREEN_W;
   h = SCREEN_H;
   for (y=0 ; y<h ; y++) {
      for (x=0 ; x<(w>>1) ; x++) {
	 temp1 = src[src_i];
	 temp2 = src[src_i+1];
	 temp1 = ((temp1 & 0xF8)>>3) + ((temp1 & 0xFC00)>>5) + ((temp1 & 0xF80000)>>8);
	 temp2 = ((temp2 & 0xF8)<<13) + ((temp2 & 0xFC00)<<11) + ((temp2 & 0xF80000)<<8);
	 dest[dest_i] = temp1 + temp2;
	 src_i+=2;
	 dest_i++;
      }
      src_i += (src_desc.lPitch>>2)-w;
      dest_i += (dest_desc.lPitch>>2)-(w>>1);
   }*/

   IDirectDrawSurface_Unlock(pseudo_screen, NULL);
   IDirectDrawSurface_Unlock(dd_prim_surface, NULL);
   exit:
   _exit_gfx_critical();
}



/* gfx_directx_create_video_bitmap_win:
 */
static struct BITMAP *gfx_directx_create_video_bitmap_win(int width, int height)
{
   LPDIRECTDRAWSURFACE surf;

   /* create DirectDraw surface */
   surf = gfx_directx_create_surface(width, height, _color_depth, 0, 0, 0);
   if (surf == NULL)
      return NULL;

   /* create Allegro bitmap for surface */
   return make_directx_bitmap(surf, width, height, _color_depth, BMP_ID_VIDEO);
}



/* gfx_directx_destroy_video_bitmap_win:
 */
static void gfx_directx_destroy_video_bitmap_win(struct BITMAP *bitmap)
{
   if (BMP_EXTRA(bitmap))
      gfx_directx_destroy_surf(BMP_EXTRA(bitmap)->surf);
   release_directx_bitmap(bitmap);
   bitmap->id &= ~BMP_ID_VIDEO;
   destroy_bitmap(bitmap);
}


/* wddwin_switch_out:
 *  handle window switched out
 */
void wddwin_switch_out(void)
{
   if (app_active) {
      app_active = FALSE;
      remove_int(update_window);
   }
}



/* wddwin_switch_in:
 *  handle window switched in
 */
void wddwin_switch_in(void)
{
   handle_window_size_win();
   if (!app_active) {
      app_active = TRUE;
      setup_retrace_proc();
   }
}


/* setup_retrace_proc
 */
static void setup_retrace_proc (void)
{ 
   HRESULT hr;
   int freq;

   /* set correct sync timer speed */
   hr = IDirectDraw_GetMonitorFrequency(directdraw, &freq);
   if ((FAILED(hr)) || (freq < 40) || (freq > 200))
      if (same_color_depth)
	 install_int_ex (update_window, BPS_TO_TIMER(70));
      else
	 install_int_ex (update_window, BPS_TO_TIMER(25));
   else
      if (same_color_depth)
	 install_int_ex (update_window, BPS_TO_TIMER(freq));
      else
	 install_int_ex (update_window, BPS_TO_TIMER(freq/3));
}


/* wnd_set_windowed_coop
 */
static int wnd_set_windowed_coop(void)
{
   HRESULT hr;

   hr = IDirectDraw_SetCooperativeLevel(directdraw, allegro_wnd, DDSCL_NORMAL);
   if (FAILED(hr)) {
      _TRACE("SetCooperative level = %s (%x), hwnd = %x\n", win_err_str(hr), hr, allegro_wnd);
      return -1;
   }

   return 0;
}

/* verify_color_depth:
 * compares the color depth requested with the real color depth
 */
static int verify_color_depth (int color_depth)
{
   DDSURFACEDESC surf_desc;
   HRESULT hr;

   /* get current video mode */
   surf_desc.dwSize = sizeof(surf_desc);
   hr = IDirectDraw_GetDisplayMode(directdraw, &surf_desc);
   if (FAILED(hr)) {
      _TRACE("Can't get color format.\n");
      return -1;
   }
   desktop_depth = surf_desc.ddpfPixelFormat.dwRGBBitCount;
   switch (desktop_depth) {
      case 8:  /* colors will be incorrect */
	 if (color_depth == 8) {
	    update_window = update_window_hw;
	    same_color_depth = TRUE;
	    break;
	 }
	 return -1;  /* no conversion from 8 bit to true color */
      case 16:
	 if (color_depth == 15 || color_depth == 16) {
	    update_window = update_window_hw;
	    same_color_depth = TRUE;
	 }
	 else if (color_depth == 32) {
	    update_window = update_window_32_to_16;
	    same_color_depth = FALSE;
	 }
	 else if (color_depth == 8) {
	    //update_window = update_window_8_to_16;
	    same_color_depth = FALSE;
	 }
	 break;
      case 32:
	 if (color_depth == 32) {
	    update_window = update_window_hw;
	    same_color_depth = TRUE;
	 }
	 else if (color_depth == 16 || color_depth == 15) {
	    //update_window = update_window_16_to_32;
	    same_color_depth = FALSE;
	 }
	 else if (color_depth == 8) {
	    //update_window = update_window_8_to_32;
	    same_color_depth = FALSE;
	 }
   }
   return 0; 
}


/* gfx_directx_show_video_bitmap_win:
 */
static int gfx_directx_show_video_bitmap_win(struct BITMAP *bitmap)
{
   if (BMP_EXTRA(bitmap)->surf) {
      pseudo_screen = BMP_EXTRA(bitmap)->surf;
      return 0;
   }
   return -1;
}

/* _get_color_shift:
 *  return shift value for color mask
 */
static int _get_color_shift(int mask)
{
   int n;

   for (n = 0; ((mask & 1) == 0) && (mask != 0); n++)
      mask >>= 1;

   return n;
}


/* create_offscreen:
 */
static void create_offscreen(int w, int h, int color_depth)
{
   DDSURFACEDESC surf_desc;
   HRESULT hr;
   int i;
   int shift_r, shift_g, shift_b;

   /* describe surface characteristics */
   surf_desc.dwSize = sizeof(surf_desc);
   surf_desc.dwFlags = DDSD_CAPS | DDSD_WIDTH | DDSD_HEIGHT;
   surf_desc.dwHeight = h;
   surf_desc.dwWidth = w;
   if (same_color_depth) {
      surf_desc.ddsCaps.dwCaps = DDSCAPS_VIDEOMEMORY;
      hr = IDirectDraw_CreateSurface(directdraw, &surf_desc, &dd_offscreen, NULL);
      if (FAILED(hr)) dd_offscreen = NULL;
      else gfx_directx_update_color_format(color_depth);
   } 
   else {
      surf_desc.dwFlags |= DDSD_PIXELFORMAT;
      /* if not the same color depth as the primary, must be in system memory (don't ask me) */
      surf_desc.ddsCaps.dwCaps = DDSCAPS_SYSTEMMEMORY | DDSCAPS_OFFSCREENPLAIN;
      for (i=0 ; pixel_match[i] ; i++) {
	 if (pixel_match[i] == color_depth) {
	    surf_desc.ddpfPixelFormat = pixel_format[i];
	    hr = IDirectDraw_CreateSurface(directdraw, &surf_desc, &dd_offscreen, NULL);
	    if (!FAILED(hr)) break;
	 }
      }
      if (pixel_match[i] == 0) {
	 _TRACE ("No se ha podido crear dd_offscreen\n");
	 dd_offscreen = NULL;
      }
      else {
	 /* update color format */
	 dd_pixelformat = &(pixel_format[i]);
	 shift_r = _get_color_shift(surf_desc.ddpfPixelFormat.dwRBitMask);
	 shift_g = _get_color_shift(surf_desc.ddpfPixelFormat.dwGBitMask);
	 shift_b = _get_color_shift(surf_desc.ddpfPixelFormat.dwBBitMask);
	 switch (color_depth) {
	    case 15:
	       _rgb_r_shift_15 = shift_r;
	       _rgb_g_shift_15 = shift_g;
	       _rgb_b_shift_15 = shift_b;
	       break;

	    case 16:
	       _rgb_r_shift_16 = shift_r;
	       _rgb_g_shift_16 = shift_g;
	       _rgb_b_shift_16 = shift_b;
	       break;

	    case 24:
	       _rgb_r_shift_24 = shift_r;
	       _rgb_g_shift_24 = shift_g;
	       _rgb_b_shift_24 = shift_b;
	       break;

	    case 32:
	       _rgb_r_shift_32 = shift_r;
	       _rgb_g_shift_32 = shift_g;
	       _rgb_b_shift_32 = shift_b;
	       break;
	 }
      }
   }
}

/* init_directx_win:
 */
static struct BITMAP *init_directx_win(int w, int h, int v_w, int v_h, int color_depth)
{
   RECT win_size;
   HRESULT hr;

   /* Flipping is impossible in windowed mode */
   if ((v_w != w && v_w != 0) || (v_h != h && v_h != 0))
      return NULL;

   _enter_critical();

   /* init DirectX */
   if (init_directx() != 0)
      goto Error;
   if (verify_color_depth(color_depth))
      goto Error;
   if (wnd_call_proc(wnd_set_windowed_coop) != 0)
      goto Error;
   if (finalize_directx_init() != 0)
      goto Error;

   /* adjust window */
   wnd_paint_back = TRUE;
   win_size.left = wnd_x = 50;
   win_size.right = 50 + w;
   win_size.top = wnd_y = 50;
   win_size.bottom = 50 + h;
   wnd_width = w;
   wnd_height = h;
   window_rect = win_size;
   wnd_windowed = TRUE;
   set_display_switch_mode(SWITCH_BACKGROUND);

   AdjustWindowRect(&win_size, GetWindowLong(allegro_wnd, GWL_STYLE), FALSE);
   MoveWindow(allegro_wnd, win_size.left, win_size.top,
   win_size.right - win_size.left, win_size.bottom - win_size.top, TRUE);

   /* create primary surface */
   if (create_primary(w, h, color_depth) != 0)
      goto Error;

   /* create offscreen backbuffer */
   create_offscreen (w, h, color_depth);
   if (dd_offscreen == NULL)
      goto Error;

   /* create clipper */
   if (create_clipper(allegro_wnd) != 0)
      goto Error;
   hr = IDirectDrawSurface_SetClipper(dd_prim_surface, dd_clipper);
   if (FAILED(hr))
      goto Error;

   if (color_depth == 8)
      if (create_palette(dd_prim_surface) != 0)
	 goto Error;

   /* setup Allegro gfx driver */
   if (!same_color_depth) {  /* use system bitmaps instead of video */
      gfx_directx_win.create_video_bitmap = gfx_directx_create_video_bitmap_win;
      gfx_directx_win.destroy_video_bitmap = gfx_directx_destroy_video_bitmap_win;
   }
   if (setup_driver(&gfx_directx_win, w, h, color_depth) != 0)
      goto Error;
   dd_frontbuffer = make_directx_bitmap(dd_offscreen, w, h, color_depth, 
					same_color_depth ? BMP_ID_VIDEO : BMP_ID_SYSTEM);

   enable_acceleration(&gfx_directx_win);

   app_active = TRUE;

   setup_retrace_proc();

   _exit_critical();

   pseudo_screen = dd_offscreen;
   return dd_frontbuffer;

 Error:
   _exit_critical();

   /* release the DirectDraw object */
   gfx_directx_exit(NULL);

   return NULL;
}

/* gfx_directx_exit:
 */
static void gfx_directx_win_exit(struct BITMAP *b)
{ 
   if (app_active) {
      app_active = FALSE;
      remove_int(update_window);
   }

   dd_pixelformat = NULL;

   if (b)
      clear (b);

   /* destroy the offscreen backbuffer used in windowed mode */
   if (dd_offscreen) {
      IDirectDrawSurface_Release(dd_offscreen);
      dd_offscreen = NULL;
   }

   /* unlink surface from bitmap */
   if (b)
      b->extra = NULL;

   gfx_directx_exit(NULL);
}



